const express = require("express");
const session = require("express-session");
const cookieParser = require("cookie-parser");
const csrf = require("csurf");
const { appSessionCheck } = require("./routers/utils");

const api = require("./routers/api");
const wellKnown = require("./routers/well-known");
const app = express();
const port = process.env.PORT || 3000;

app.locals.FORBIDDEN_USERNAMES = [ "__proto__", "prototype", "constructor" ];
app.locals.USERS = [{
	id: crypto.randomUUID(),
	username: process.env.BOT_USERNAME,
	email: `${process.env.BOT_USERNAME}@molecule.fr`,
	password: process.env.BOT_PASSWORD,
	secret: process.env.FLAG,
	approved_clients: []
}]

app.use((req, res, next) => {
	if (req.headers.host !== process.env.IDP_DOMAIN.split("//")[1]) return res.redirect(302, process.env.IDP_DOMAIN);
	next();
});

app.set("view engine", "ejs");
app.set("views", "./views");

const csrfMiddleware = csrf({ cookie: { httpOnly: true, secure: true, sameSite: "None"} });
app.use(express.json({ type: ["application/json", "text/*"] }));
app.use(cookieParser());
app.use((req, res, next) => {
	if (!req.cookies["cacheId"]) {
		res.locals.cacheId = crypto.randomUUID();
		res.cookie("cacheId", res.locals.cacheId, {
			httpOnly: true,
			secure: true,
			sameSite: "none",
			maxAge: 1000 * 60 * 60 * 24 // 1 day
		});
	}
	next();
});

app.use(session({
	secret: crypto.randomUUID(),
	resave: true,
	saveUninitialized: false,
	cookie: {
		httpOnly: true,
		sameSite: "Lax",
		maxAge: 1000 * 60 * 60 * 24 // 1 day
	}
}));

app.use((req, res, next) => {
	if (req.headers["x-csrf-token"]) res.set("X-CSRF-Token", req.headers["x-csrf-token"])
	next();
});

app.use("/public", express.static("public"));
app.use("/.well-known", wellKnown);
app.use("/api", api);

app.get("/", csrfMiddleware, (req, res) => {
	if (req.session.username) return res.redirect(307, "/profile");
	res.render("login", { csrfToken: req.csrfToken() });
});

app.get("/profile", appSessionCheck, csrfMiddleware, (req, res) => {
	const user = app.locals.USERS.filter((user) => user.username === req.session.username)[0];
	if (!user) return res.redirect(307, "/");
	res.render("profile", { csrfToken: req.csrfToken() });
});

app.get("/authorization", appSessionCheck, csrfMiddleware, (req, res) => {
	if (!req.session.username) return res.redirect(307, "/");

	const scope = (req.query.scope || "").split(",");
	res.render("authorization", {
		csrfToken: req.csrfToken(),
		rp_url: req.query.rp_url,
		id: !!scope.includes("id"),
		username: !!scope.includes("username"),
		secret: !!scope.includes("secret")
	});
});

app.use((err, req, res, next) => {
	if (err.code === "EBADCSRFTOKEN") {
		return res.status(403).json({
			error: "Invalid CSRF token. Please try again.",
			message: err.message
		});
	}
	next(err);
});

const listener = app.listen(port, () => {
	console.log("Your app is listening on port " + listener.address().port);
});
