const { csrfCheck, apiSessionCheck } = require("./utils");
const express = require("express");

const router = express.Router();
router.use(express.urlencoded({ extended: true }));

// https://developers.google.com/privacy-sandbox/cookies/fedcm/implement/identity-provider#idp-config-file
router.get("/accounts", csrfCheck, apiSessionCheck, (req, res) => {
	const user = res.locals.user;
	return res.json({
		accounts: [{
			id: user.id,
			name: user.username,
			email: user.email,
			picture: `${process.env.IDP_DOMAIN}public/img/profile.png`,
			approved_clients: user.approved_clients
		}]
	});
})

router.get("/metadata", (req, res) => {
	return res.json({
		privacy_policy_url: process.env.IDP_DOMAIN,
		terms_of_service_url: process.env.IDP_DOMAIN,
		icons: [{
			url: `${process.env.IDP_DOMAIN}/public/img/favicon.svg`,
			size: 40
		}]
	});
});

var RP_CLIENT  = {};
RP_CLIENT[process.env.APP_DOMAIN] = { "client_secret": process.env.CLIENT_SECRET }
const ALLOWED_CODE = new Map();
const ALLOWED_ORIGIN = [ process.env.APP_DOMAIN, process.env.IDP_DOMAIN ];
const getCode = (user, scope) => {
	var accessibleData = {};
	const code = crypto.randomUUID();

	if (scope) {
		for (const attr of scope) {
			accessibleData[attr] = user[attr];
		}
	} else {
		accessibleData = {
			id: user.id,
			name: user.username,
			email: user.email
		}
	}

	ALLOWED_CODE.set(code, accessibleData);
	return code;
}

router.post("/token", csrfCheck, apiSessionCheck, (req, res) => {
	const { client_id, scope } = req.body;
	const user = res.locals.user;
	const reqOrigin = new URL(req.headers.origin).origin;

	if (
		!Object.keys(RP_CLIENT).includes(client_id) ||
		!ALLOWED_ORIGIN.includes(reqOrigin)
	) {
		return res.status(400).json({ error: "Invalid request." });
	}

	return res.json({ token: getCode(user, scope.split(",")) });
});

router.post("/idtokens", csrfCheck, apiSessionCheck, (req, res) => {
	const {
		client_id,
		account_id,
		consent_acquired,
		disclosure_text_shown,
		params,
	} = req.body;

	const paramsObject = JSON.parse(params || "{}");
	const user = res.locals.user;
	const reqOrigin = new URL(req.headers.origin).origin;

	if (
		!Object.keys(RP_CLIENT).includes(client_id) ||
		!ALLOWED_ORIGIN.includes(reqOrigin) ||
		account_id !== user.id
	) {
		return res.status(400).json({ error: "Invalid request." });
	}

	if (user.approved_clients.includes(reqOrigin)) return res.json({ token: getCode(user, paramsObject.scope) });
	if (paramsObject.scope) {
		return res.json({
			continue_on: `/authorization?client_id=${client_id}&scope=${paramsObject.scope}`,
		});
	}

	if (
		(consent_acquired === "true" || disclosure_text_shown === "true" || is_auto_selected === "true") && 
		!user.approved_clients.includes(reqOrigin)
	) {
		user.approved_clients.push(reqOrigin);
	}

	return res.json({ token: getCode(user, paramsObject.scope) });
});

router.post("/data", (req, res) => {
	const { code, client_id, client_secret } = req.body;

	if (!RP_CLIENT[client_id] || client_secret !== RP_CLIENT[client_id].client_secret) return res.status(401).json({ error: "Invalid client_secret." });
	if (!ALLOWED_CODE.has(code)) return res.status(401).json({ error: "Invalid code." });

	return res.json(ALLOWED_CODE.get(code));
});

router.post("/disconnect", csrfCheck, apiSessionCheck, (req, res) => {
	const { account_hint, client_id } = req.body;
	const user = res.locals.user;

	if (user.id !== account_hint && !Object.keys(RP_CLIENT).includes(client_id)) {
		return res.status(400).json({ error: "Invalid request." });
	}

	user.approved_clients = user.approved_clients.filter(
		(client) => client !== client_id
	);
	return res.json({ account_id: user.id });
});

module.exports = router;