import { getToken, clearToken } from "../utils/tokenStore";
const API_BASE_URL = "/api";

const getAuthHeaders = () => {
	const token = getToken("bearerToken");
	return {
		"Content-Type": "application/json",
		...(token ? { "Authorization": `Bearer ${token}` } : {})
	};
};

const handleResponse = async (response) => {
	if (!response.ok) {
		if (response.status === 401) {
			clearToken("bearerToken");
			window.location.reload();
			throw new Error("Authentication expired");
		}
		
		const text = await response.text();
		try {
			const json = JSON.parse(text);
			throw new Error(json.error || "API Error");
		} catch (e) {
			throw new Error(`API Error: ${text}`);
		}
	}
	
	const text = await response.text();
	try {
		return text ? JSON.parse(text) : {};
	} catch (e) {
		throw new Error("Invalid JSON response");
	}
};

export const api = {
	// Projects
	getProjects: async () => {
		try {
			const response = await fetch(`${API_BASE_URL}/projects`, {
				headers: getAuthHeaders()
			});
			return handleResponse(response);
		} catch (error) {
			console.error("API Error:", error);
			return [];
		}
	},

	createProject: async (name) => {
		const response = await fetch(`${API_BASE_URL}/projects`, {
			method: "POST",
			headers: getAuthHeaders(),
			body: JSON.stringify({ name }),
		});
		return handleResponse(response);
	},

	updateProject: async (projectId, name) => {
		const response = await fetch(`${API_BASE_URL}/projects/${projectId}`, {
			method: "PUT",
			headers: getAuthHeaders(),
			body: JSON.stringify({ name }),
		});
		return handleResponse(response);
	},

	deleteProject: async (projectId) => {
		const response = await fetch(`${API_BASE_URL}/projects/${projectId}`, {
			method: "DELETE",
			headers: getAuthHeaders(),
		});
		return handleResponse(response);
	},

	// Files
	createFile: async (projectId, { name, content = "", type }) => {
		const response = await fetch(`${API_BASE_URL}/projects/${projectId}/files`, {
			method: "POST",
			headers: getAuthHeaders(),
			body: JSON.stringify({ name, content, type }),
		});
		return handleResponse(response);
	},

	updateFile: async (projectId, fileId, updates) => {
		const response = await fetch(
			`${API_BASE_URL}/projects/${projectId}/files/${fileId}`,
			{
				method: "PUT",
				headers: getAuthHeaders(),
				body: JSON.stringify(updates),
			}
		);
		return handleResponse(response);
	},

	deleteFile: async (projectId, fileId) => {
		const response = await fetch(
			`${API_BASE_URL}/projects/${projectId}/files/${fileId}`,
			{
				method: "DELETE",
				headers: getAuthHeaders(),
			}
		);
		return handleResponse(response);
	},

	saveFileContent: async (projectId, fileId, content) => {
		return await api.updateFile(projectId, fileId, { content });
	}
};