import { useState, useEffect } from "react";
import { api } from "../services/api";
import { getFileType } from "../utils/fileUtils";

export const useProjectOperations = (initialProjects = [], {
	handleTabClose,
	openTabs
} = {}) => {
	const [projects, setProjects] = useState(initialProjects);
	const [loading, setLoading] = useState(true);
	const [error, setError] = useState(null);

	useEffect(() => {
		loadProjects();
	}, []);

	const loadProjects = async () => {
		try {
			setLoading(true);
			const data = await api.getProjects();
			setProjects(data);
			setError(null);
		} catch (err) {
			setError("Failed to load projects");
			console.error("Error loading projects:", err);
		} finally {
			setLoading(false);
		}
	};

	const addProject = async (name) => {
		if (!name.trim()) return;
		try {
			const newProject = await api.createProject(name);
			setProjects([...projects, newProject]);
			return newProject;
		} catch (err) {
			setError("Failed to create project");
			console.error("Error creating project:", err);
		}
	};

	const deleteProject = async (projectId) => {
		try {
			if (openTabs && handleTabClose) {
				const projectTabs = openTabs.filter(tab => tab.projectId === projectId);
				projectTabs.forEach(tab => {
					handleTabClose(tab.id);
				});
			}
			
			await api.deleteProject(projectId);
			setProjects(projects.filter(p => p.id !== projectId));
		} catch (err) {
			setError("Failed to delete project");
			console.error("Error deleting project:", err);
		}
	};

	const addFile = async (projectId, fileName) => {
		if (!fileName.trim()) return null;
		try {
			const newFile = await api.createFile(projectId, {
				name: fileName,
				type: getFileType(fileName)
			});
			
			setProjects(projects.map(project => {
				if (project.id === projectId) {
					return {
						...project,
						files: [...project.files, newFile]
					};
				}
				return project;
			}));
		} catch (err) {
			setError("Failed to create file");
			console.error("Error creating file:", err);
		}
	};

	const deleteFile = async (projectId, fileId) => {
		try {
			if (openTabs && handleTabClose) {
				const tabId = `${projectId}||${fileId}`;
				const existingTab = openTabs.find(tab => tab.id === tabId);
				if (existingTab) {
					handleTabClose(tabId);
				}
			}

			await api.deleteFile(projectId, fileId);
			
			setProjects(projects.map(project => {
				if (project.id === projectId) {
					return {
						...project,
						files: project.files.filter(f => f.id !== fileId)
					};
				}
				return project;
			}));
		} catch (err) {
			setError("Failed to delete file");
			console.error("Error deleting file:", err);
		}
	};

	const renameProject = async (projectId, newName) => {
		if (!newName.trim()) return;
		try {
			const updatedProject = await api.updateProject(projectId, newName);
			setProjects(projects.map(project => 
				project.id === projectId ? updatedProject : project
			));
		} catch (err) {
			setError("Failed to rename project");
			console.error("Error renaming project:", err);
		}
	};

	const renameFile = async (projectId, fileId, newName) => {
		if (!newName.trim()) return;
		try {
			const updatedFile = await api.updateFile(projectId, fileId, {
				name: newName,
				type: getFileType(newName)
			});
			
			setProjects(projects.map(project => {
				if (project.id === projectId) {
					return {
						...project,
						files: project.files.map(file => 
						file.id === fileId ? updatedFile : file
						)
					};
				}
				return project;
			}));
		} catch (err) {
			setError("Failed to rename file");
			console.error("Error renaming file:", err);
		}
	};

	const saveFileContent = async (projectId, fileId, content) => {
		try {
			const updatedFile = await api.updateFile(projectId, fileId, { content });
			return updatedFile;
		} catch (err) {
			setError("Failed to save file");
			console.error("Error saving file:", err);
			throw err;
		}
	};

	return {
		projects,
		setProjects,
		addProject,
		deleteProject,
		addFile,
		deleteFile,
		renameProject,
		renameFile,
		saveFileContent,
		loading,
		error
	};
};