import { useState, useEffect } from "react";
import { useProjectOperations } from "./useProjectOperations";
import { setFileInUrl, getFileFromUrl } from "../utils/urlUtils";

const DEFAULT_PROJECTS = [];

export const useFileSystem = () => {
	const [fileContents, setFileContents] = useState({});
	const [openTabs, setOpenTabs] = useState([]);
	const [activeTab, setActiveTab] = useState(null);
	const [unsavedChanges, setUnsavedChanges] = useState({});
	const [lastSaved, setLastSaved] = useState({});
	const [previewMode, setPreviewMode] = useState({});

	const handleTabClose = (tabId) => {
		if (!tabId) return;

		if (unsavedChanges[tabId]) {
			const confirmed = window.confirm("You have unsaved changes. Do you want to close this tab?");
			if (!confirmed) return;
		}
		
		if (activeTab === tabId) {
			setFileInUrl(null, null);
		}
		
		setOpenTabs(openTabs.filter(tab => tab.id !== tabId));
		if (activeTab === tabId) {
			const remainingTabs = openTabs.filter(tab => tab.id !== tabId);
			if (remainingTabs.length > 0) {
				setActiveTab(remainingTabs[0].id);
				const [projectId, fileId] = remainingTabs[0].id.split("||");
				const shouldRender = previewMode[remainingTabs[0].id] || false;
				setFileInUrl(projectId, fileId, shouldRender);
			} else {
				setActiveTab(null);
			}
		}

		const newUnsavedChanges = { ...unsavedChanges };
		delete newUnsavedChanges[tabId];
		setUnsavedChanges(newUnsavedChanges);

		const newFileContents = { ...fileContents };
		delete newFileContents[tabId];
		setFileContents(newFileContents);

		const newPreviewMode = { ...previewMode };
		delete newPreviewMode[tabId];
		setPreviewMode(newPreviewMode);
	};

	const {
		projects,
		setProjects,
		addProject,
		deleteProject,
		addFile,
		deleteFile,
		renameProject,
		renameFile,
		saveFileContent,
		loading,
		error
	} = useProjectOperations(DEFAULT_PROJECTS, {
		handleTabClose,
		openTabs,
		setOpenTabs,
		setActiveTab
	});

	useEffect(() => {
		const handleUrlChange = () => {
		const fileInfo = getFileFromUrl();
		if (fileInfo && projects && projects.length > 0) {
			const { projectId, fileId, render } = fileInfo;
			const project = projects.find(p => p.id === projectId);
			if (project) {
				const file = project.files.find(f => f.id === fileId);
				if (file) {
					const tabId = `${projectId}||${fileId}`;

					if (render && file.type === "md") {
						setPreviewMode(prev => ({
							...prev,
							[tabId]: true
						}));
					}

					if (!openTabs.find(tab => tab.id === tabId)) {
						setOpenTabs([...openTabs, {
							id: tabId,
							projectId,
							file,
						}]);
						setFileContents(prev => ({
							...prev,
							[tabId]: file.content
						}));
					}
					setActiveTab(tabId);
				}
			}
		}
		};

		window.addEventListener("popstate", handleUrlChange);
		if (projects && !loading) {
			handleUrlChange();
		}

		return () => {
			window.removeEventListener("popstate", handleUrlChange);
		};
	}, [projects, loading]);

	const handleSave = async (tabId) => {
		if (!unsavedChanges[tabId]) return;

		try {
			const [projectId, fileId] = tabId.split("||");
			const content = fileContents[tabId];

			const updatedFile = await saveFileContent(projectId, fileId, content);

			setProjects(projects.map(project => {
				if (project.id === projectId) {
					return {
						...project,
						files: project.files.map(file => {
							if (file.id === fileId) {
								return updatedFile;
							}
							return file;
						})
					};
				}
				return project;
			}));

			setUnsavedChanges(prev => ({
				...prev,
				[tabId]: false
			}));
			setLastSaved(prev => ({
				...prev,
				[tabId]: new Date().toLocaleTimeString()
			}));
		} catch (error) {
			console.error("Failed to save file:", error);
		}
	};

	const handleFileOpen = (projectId, file) => {
		if (!projectId || !file) return;

		const tabId = `${projectId}||${file.id}`;
		if (!openTabs.find(tab => tab.id === tabId)) {
			setOpenTabs([...openTabs, {
				id: tabId,
				projectId,
				file,
			}]);
			setFileContents(prev => ({
				...prev,
				[tabId]: file.content
			}));
		}
		setActiveTab(tabId);

		const urlParams = new URLSearchParams(window.location.search);
		const shouldRender = urlParams.get("render") === "true";

		setFileInUrl(projectId, file.id, shouldRender || previewMode[tabId] || false);
	};

	const handleContentUpdate = (tabId, newContent) => {
		setFileContents(prev => ({
			...prev,
			[tabId]: newContent
		}));
		setUnsavedChanges(prev => ({
			...prev,
			[tabId]: true
		}));
	};

	const handlePreviewToggle = (tabId) => {
		const newPreviewMode = !previewMode[tabId];
		setPreviewMode(prev => ({
			...prev,
			[tabId]: newPreviewMode
		}));

		const tab = openTabs.find(t => t.id === tabId);
		if (tab) {
			const [projectId, fileId] = tabId.split("||");
			setFileInUrl(projectId, fileId, newPreviewMode);
		}
	};

	return {
		projects: projects || [],
		openTabs,
		activeTab,
		setActiveTab,
		fileContents,
		unsavedChanges,
		lastSaved,
		previewMode,
		handleSave,
		handleFileOpen,
		handleTabClose,
		handleContentUpdate,
		handlePreviewToggle,
		addProject,
		deleteProject,
		addFile,
		deleteFile,
		renameProject,
		renameFile,
		loading,
		error
	};
};