import React from "react";
import { Folder, FileText, Plus, Edit3, Trash2 } from "lucide-react";
import IconButton from "../shared/IconButton";

const FileTree = ({
	projects,
	showNewFileInput,
	editingName,
	onFileOpen,
	onNewFile,
	onDeleteProject,
	onDeleteFile,
	onRenameProject,
	onRenameFile,
	setShowNewFileInput,
	setEditingName,
}) => {
	return (
		<>
		{projects.map(project => (
			<div key={project.id} className="mb-4">
				<div className="flex items-center justify-between mb-2">
					<div className="flex items-center flex-1">
						<Folder size={16} className="mr-2 flex-shrink-0" />
						{editingName.type === "project" && editingName.id === project.id ? (
							<form 
								onSubmit={(e) => {
									e.preventDefault();
									const input = e.target.elements.name;
									onRenameProject(project.id, input.value);
									setEditingName({ type: null, id: null });
								}}
								className="flex-1"
							>
							<input
								name="name"
								type="text"
								defaultValue={project.name}
								className="bg-gray-700 text-white px-2 py-1 rounded w-full"
								onBlur={(e) => {
									onRenameProject(project.id, e.target.value);
									setEditingName({ type: null, id: null });
								}}
								autoFocus
							/>
							</form>
						) : (
							<span className="flex-1">{project.name}</span>
						)}
					</div>
					<div className="flex items-center">
						<IconButton
							onClick={() => setEditingName({ type: "project", id: project.id })}
							icon={<Edit3 size={16} />}
							title="Rename project"
							size="sm"
							variant="ghost"
							className="mr-1"
						/>
						<IconButton
							onClick={() => onDeleteProject(project.id)}
							icon={<Trash2 size={16} />}
							title="Delete project"
							size="sm"
							variant="danger"
							className="mr-1"
						/>
						<IconButton
							onClick={() => setShowNewFileInput(project.id)}
							icon={<Plus size={16} />}
							title="Add file"
							size="sm"
							variant="ghost"
						/>
					</div>
				</div>
				
				<div className="pl-6">
					{showNewFileInput === project.id && (
						<div className="flex items-center mb-2">
							<input
							type="text"
							placeholder="filename.ext"
							className="flex-1 bg-gray-700 text-white px-2 py-1 rounded text-sm"
							onKeyPress={(e) => {
								if (e.key === "Enter") {
									onNewFile(project.id, e.target.value);
								}
							}}
							onBlur={(e) => onNewFile(project.id, e.target.value)}
							autoFocus
							/>
						</div>
					)}
					{project.files.map(file => (
						<div
							key={file.id}
							className="flex items-center py-1 px-2 hover:bg-gray-700 rounded group"
						>
							<div 
								className="flex items-center flex-1 cursor-pointer"
								onClick={() => onFileOpen(project.id, file)}
							>
							<FileText size={16} className="mr-2 flex-shrink-0" />
							{editingName.type === "file" && editingName.id === file.id ? (
								<form 
									onSubmit={(e) => {
										e.preventDefault();
										const input = e.target.elements.name;
										onRenameFile(project.id, file.id, input.value);
										setEditingName({ type: null, id: null });
									}}
									className="flex-1"
								>
								<input
									name="name"
									type="text"
									defaultValue={file.name}
									className="bg-gray-700 text-white px-2 py-1 rounded w-full"
									onBlur={(e) => {
										onRenameFile(project.id, file.id, e.target.value);
										setEditingName({ type: null, id: null });
									}}
									autoFocus
								/>
								</form>
							) : (
								<span>{file.name}</span>
							)}
							</div>
							<div className="flex items-center opacity-0 group-hover:opacity-100 transition-opacity">
								<IconButton
									onClick={(e) => {
										e.stopPropagation();
										setEditingName({ type: "file", id: file.id });
									}}
									icon={<Edit3 size={14} />}
									title="Rename file"
									size="sm"
									variant="ghost"
									className="mr-1"
								/>
								<IconButton
									onClick={(e) => {
										e.stopPropagation();
										onDeleteFile(project.id, file.id);
									}}
									icon={<Trash2 size={14} />}
									title="Delete file"
									size="sm"
									variant="danger"
								/>
							</div>
						</div>
					))}
				</div>
			</div>
		))}
		</>
	);
};

export default FileTree;