const express = require("express");
const jwt = require("jsonwebtoken");
const router = express.Router();

const JWT_SECRET = crypto.randomUUID();
const authenticateJWT = (req, res, next) => {
	const authHeader = req.headers.authorization;

	if (!authHeader || !authHeader.startsWith("Bearer ")) {
		return res.status(401).json({ error: "Unauthorized" });
	}

	const token = authHeader.split(" ")[1];

	try {
		const user = jwt.verify(token, JWT_SECRET);

		// I don't want players to be able to update the bot's projects
		if (user.username === process.env.BOT_USERNAME && req.method !== "GET") {
			return res.status(401).json({ error: "Invalid token" });
		}

		req.user = user;
		next();
	} catch (error) {
		return res.status(401).json({ error: "Invalid token" });
	}
};

router.get("/idp", (req, res) => {
	const idpUrl = process.env.IDP_DOMAIN;
	const nonce = crypto.randomUUID();
	req.session.authNonce = nonce;

	res.json({
		idpUrl: idpUrl,
		idpNonce: nonce,
		configURL: `${idpUrl}/.well-known/fedcm.json`,
		clientId: process.env.APP_DOMAIN,
		scope: ["id", "username", "secret"]
	});
});

router.post("/callback", async (req, res) => {
	const { code, nonce } = req.body;

	if (nonce !== req.session.authNonce) {
		return res.status(400).json({ error: "Invalid nonce" });
	}

	try {
		const idpUrl = process.env.IDP_DOMAIN;
		const response = await fetch(`${idpUrl}/api/fedcm/data`, {
			method: "POST",
			headers: {
				"Content-Type": "application/json"
			},
			body: JSON.stringify({ 
				code,
				client_id: process.env.APP_DOMAIN,
				client_secret: process.env.CLIENT_SECRET
			})
		});

		if (!response.ok) {
			throw new Error("Failed to exchange code for token");
		}

		const userData = await response.json();
		const token = jwt.sign({
				client_id: process.env.APP_DOMAIN,
				code: code,
				sub: userData.id,
				username: userData.username,
			},
			JWT_SECRET,
			{ expiresIn: "1h" }
		);

		req.session.authNonce = null;
		return res.json({ username: userData.username, token });

	} catch (error) {
		console.error("Auth callback error:", error);
		return res.status(500).json({ error: "Authentication failed" });
	}
});

router.get("/verify", authenticateJWT, (req, res) => {
	res.json({ 
		valid: true, 
		user: req.user,
		message: "Token is valid" 
	});
});

module.exports = { router, authenticateJWT };