const express = require("express");
const crypto = require("crypto");
const router = express.Router();

const userProjects = new Map();
const getProjectsForUser = (userId) => {
	if (!userProjects.has(userId)) {
		userProjects.set(userId, [
			{
				id: crypto.randomUUID(),
				name: "Getting Started",
				files: [
					{ id: crypto.randomUUID(), name: "README.md", content: "# Welcome to Molecule Editor\nThis is your first project. Feel free to edit or create new files.", type: "md" },
					{ id: crypto.randomUUID(), name: "styles.css", content: "body {\n  color: #333;\n  font-family: 'Arial', sans-serif;\n}", type: "css" }
				]
			}
		]);
	}
	return userProjects.get(userId);
};

router.get("/", (req, res) => {
	const userId = req.user.sub;
	res.json(getProjectsForUser(userId));
});

router.post("/", (req, res) => {
	const userId = req.user.sub;
	const { name } = req.body;
	const projects = getProjectsForUser(userId);

	const newProject = {
		id: crypto.randomUUID(),
		name,
		files: []
	};

	projects.push(newProject);
	userProjects.set(userId, projects);

	res.json(newProject);
});

router.put("/:id", (req, res) => {
	const userId = req.user.sub;
	const { id } = req.params;
	const { name } = req.body;
	const projects = getProjectsForUser(userId);

	const updatedProjects = projects.map(project =>
		project.id === id ? { ...project, name } : project
	);

	userProjects.set(userId, updatedProjects);
	res.json(updatedProjects.find(p => p.id === id));
});

router.delete("/:id", (req, res) => {
	const userId = req.user.sub;
	const { id } = req.params;
	const projects = getProjectsForUser(userId);

	const filteredProjects = projects.filter(p => p.id !== id);
	userProjects.set(userId, filteredProjects);

	res.json({ success: true });
});

router.post("/:projectId/files", (req, res) => {
	const userId = req.user.sub;
	const { projectId } = req.params;
	const { name, content = "", type } = req.body;
	const projects = getProjectsForUser(userId);

	const project = projects.find(p => p.id === projectId);
	if (!project) {
		return res.status(404).json({ error: "Project not found" });
	}

	const newFile = {
		id: crypto.randomUUID(),
		name,
		content,
		type
	};

	project.files.push(newFile);
	userProjects.set(userId, projects);

	res.json(newFile);
});

router.put("/:projectId/files/:fileId", (req, res) => {
	const userId = req.user.sub;
	const { projectId, fileId } = req.params;
	const { name, content, type } = req.body;
	const projects = getProjectsForUser(userId);

	const project = projects.find(p => p.id === projectId);
	if (!project) {
		return res.status(404).json({ error: "Project not found" });
	}

	project.files = project.files.map(file => {
		if (file.id === fileId) {
			return {
				...file,
				...(name && { name }),
				...(content && { content }),
				...(type && { type })
			};
		}
		return file;
	});

	userProjects.set(userId, projects);
	res.json(project.files.find(f => f.id === fileId));
});

router.delete("/:projectId/files/:fileId", (req, res) => {
	const userId = req.user.sub;
	const { projectId, fileId } = req.params;
	const projects = getProjectsForUser(userId);

	const project = projects.find(p => p.id === projectId);
	if (!project) {
		return res.status(404).json({ error: "Project not found" });
	}

	project.files = project.files.filter(f => f.id !== fileId);
	userProjects.set(userId, projects);

	res.json({ success: true });
});

module.exports = { router, userProjects, getProjectsForUser };